"use client";

import { use, useEffect, useState } from "react";
import { notFound } from "next/navigation";
import wordpressAPI from "@/lib/wordpress";
import { useLanguage } from "@/contexts/LanguageContext";

interface PageProps {
  params: Promise<{ slug: string }>; // ✅ params is now a Promise
}

export default function PortfolioPage({ params }: PageProps) {
  const { slug } = use(params); // ✅ unwrap the Promise
  const { language } = useLanguage();
  const [portfolio, setPortfolio] = useState<any | null>(null);

  useEffect(() => {
    const fetchPortfolio = async () => {
      try {
        const data = await wordpressAPI.getPostBySlugAndType("portfolio", slug, language);
        setPortfolio(data);
      } catch (error) {
        console.error("Error loading portfolio:", error);
        setPortfolio(null);
      }
    };

    fetchPortfolio();
  }, [slug, language]);

  // Still fetching → show nothing or skeleton
  if (portfolio === null) {
    return null;
  }

  // If nothing found
  if (!portfolio) {
    notFound();
  }

  // Extract portfolio image (featured image with the new size)
  const featuredImage =
    portfolio._embedded?.["wp:featuredmedia"]?.[0]?.media_details?.sizes["single-post-featured-image"]?.source_url ||
    portfolio._embedded?.["wp:featuredmedia"]?.[0]?.source_url;

  // Extract portfolio metadata
  const author = portfolio._embedded?.author?.[0]?.name || "Unknown Author";
  const date = portfolio.date ? new Date(portfolio.date).toLocaleDateString() : "Unknown Date";
  const categories = portfolio._embedded?.["wp:term"]?.[0]?.map((cat: any) => cat.name).join(", ") || "No Categories";

  return (
    <div className="portfolio-content">
      <div className="container my-8">
        {featuredImage && (
          <div className="mb-6 relative w-full">
            <div className="aspect-[16/9] object-cover">
              <img
                src={featuredImage}
                alt={portfolio.title?.rendered || "Featured image"}
                className="rounded-xl object-cover w-full h-full"
              />
            </div>
          </div>
        )}

        <h1
          className="text-4xl font-bold mb-8 text-gray-900 dark:text-gray-100"
          dangerouslySetInnerHTML={{
            __html:
              typeof portfolio.title === "string"
                ? portfolio.title
                : portfolio.title?.rendered || "",
          }}
        />

        <div className="text-sm text-gray-500 dark:text-gray-300 mb-4">
          <span className="mr-4">By: {author}</span>
          <span className="mr-4">Published: {date}</span>
          <span>Categories: {categories}</span>
        </div>

        <div
          className="te-portfolio-post prose dark:prose-invert prose-md prose-img:rounded-xl prose-a:text-primary-600 dark:prose-a:text-primary-400 hover:prose-a:text-gray-500 max-w-none"
          dangerouslySetInnerHTML={{
            __html:
              typeof portfolio.content === "string"
                ? portfolio.content
                : portfolio.content?.rendered || "",
          }}
        />
      </div>
    </div>
  );
}
